package gov.va.med.mhv.phr.portlet;

import gov.va.med.mhv.beehive.EntityReadOnlyListPageFlowController;
import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.phr.util.PHRMessages;
import gov.va.med.mhv.phr.util.SessionManagement;
import gov.va.med.mhv.usermgmt.enumeration.ExtractType;
import gov.va.med.mhv.usermgmt.service.PatientExtractStatusServiceResponse;
import gov.va.med.mhv.usermgmt.service.PatientServiceResponse;
import gov.va.med.mhv.usermgmt.service.delegate.InPersonAuthenticationServiceDelegate;
import gov.va.med.mhv.usermgmt.service.delegate.ServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.transfer.PatientExtractStatus;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.util.PatientCorrelationStatusUtils;

import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;

import org.apache.beehive.netui.pageflow.Forward;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.messages.Message;
import org.tigris.atlas.messages.MessageFactory;
import org.tigris.atlas.messages.Messages;
import org.tigris.atlas.service.EntityCollectionServiceResponse;
import org.tigris.atlas.service.ServiceResponse;

import com.bea.portlet.PageURL;

public abstract class ExtractPageFlowBaseController
    extends EntityReadOnlyListPageFlowController
{
    private static final Log LOG = LogFactory.getLog(
        ExtractPageFlowBaseController.class);

    private static final String IPA_PORTAL_PAGE = "inPersonAuthentication";

    public static final String SHOW_NOT_IN_PERSON_AUTHENTICATED_ACTION =
        "showNotInPersonAuthenticated";
    private static final InPersonAuthenticationServiceDelegate IPA_DELEGATE = 
        ServiceDelegateFactory.createInPersonAuthenticationServiceDelegate();

    private PatientExtractStatus status = null;
    private boolean wasDisplayableOnLastLoad = false;
    private String inPersonAuthenticationUrl = null;
    private boolean patientIsCorrelatedDisplayFlag = true;

    protected boolean allowAdvancedPatient(Patient patient) {
    	return false;
    }
    
    public Forward begin() {
        Forward forward = null;
   	    //Jazz Task#39184:VA Allergies and Adverse Reactions Detail Path - Made sure to check if the user is IPA since getPatient is also valid for advanced users.
        boolean isPatientEligibleToView = true;
        if(getPatient()!=null && !allowAdvancedPatient(getPatient())) {
        	isPatientEligibleToView = isIPAedPatient(getPatient());
        } else {
        	isPatientEligibleToView = true;
        }
        
        if (getPatient() == null||!isPatientEligibleToView) {
            forward = new Forward("patient-not-ipa-ed");
            inPersonAuthenticationUrl = createInPersonAuthenticationUrl();
        } else if (!isCorrelated()) {
        		forward = new Forward("patient-not-correlated");
        } else {
            auditView();
        	//If the extract is not enabled or the extract is partialed display
        	//an info message.
            if (!isEnabled()) {
                addExtractUnavailableMessage();
            } else if (isPartialed()) {
                addExtractUpdatesUnavailableMessage();
            }
            forward = super.begin();
        }
        return forward;
    }

    /**
     * 
	 * Jazz Task#39184:VA Allergies and Adverse Reactions Detail Path - Created this method to check if the user is a Premium user
     */
    private boolean isIPAedPatient(Patient patient) {
        PatientServiceResponse serviceResponse = getIPADelegate().
            getIPAedPatientForUser(patient.getUserProfile().getUserName());
        return serviceResponse.getPatient()!=null;
    }
    
    private boolean hasNotAPatientMessage(ServiceResponse serviceResponse) {
		Messages msgs = serviceResponse.getMessages();
		for(Object errObj : msgs.getErrorMessages()) {
			Message m = (Message)errObj;
			if ( "user.not.a.patient".equals(m.getKey()) ) return true;
		}
		return false;
	}
    
    private InPersonAuthenticationServiceDelegate getIPADelegate() {
        return IPA_DELEGATE;
    }
    
    public Forward refresh() {
        Forward forward = new Forward("success");
        return forward;
    }

    public Forward showNotInPersonAuthenticated() {
        return new Forward("success");
    }

    /**
     * Get the patient from the Session. If found
     * get Status (based on if they are correlated return true)
     * or return false for anything else!
     * @return Is patient Correlated?
     */
    public boolean isCorrelated(){
    	boolean correlation = false;
    	Patient patient = getPatient();
    	//Patient check, user can be null
    	//but if they are null for a patient they
    	//are naturally not correlated, nor IPA'd. However,
    	//if they are IPA'd this can catch the correlation status.
    	if(patient != null){
    		correlation =
    			PatientCorrelationStatusUtils.isCorrelated(patient);
    	}
    	if(LOG.isInfoEnabled()){
    		LOG.info("Patient is correlated == " + correlation);
    	}
    	return correlation;
    }

    /**
     * Callback that is invoked when this controller instance is created.
     */
    @Override
    protected void onCreate() {
    }

    /* (non-Javadoc)
     * @see org.tigris.atlas.beehive.EntityMaintenancePageFlowController#
     * getCurrentUserId()
     */
    @Override
    protected Long getCurrentUserId() {
        return getUserProfile().getId();
    }

    protected Collection getEntities() {
        return (Collection) getSession().getAttribute( getTableSessionKey());
    }

    /**
     * Gets the Patient instance for the current user.
     * @return The patient.
     */
    public Patient getPatient() {
        return SessionManagement.getInstance().getPatient(getRequest());
    }
    
    /**
     * Gets the Patient id instance for the current user.
     * @return The patient.
     */
    public long getPatientId() {
        return SessionManagement.getInstance().getPatient(getRequest()).getId();
    }    

    /**
     * Gets the UserProfile instance for the current user.
     * The user must be an in-person-authenticated patient.
     * @return The patient.
     */
    public UserProfile getUserProfile() {
        return getPatient().getUserProfile();
    }
    
    /**
     * Gets the full name (including middle name) of the current patient.
     * This will be in the format: First Name M. Last Name
     * @return The patient name.
     * @see #getPatient()
     */
    public String getFullPatientName() {
		StringBuffer buffer = new StringBuffer(100);
		buffer.append("");    	
    	UserProfile up = getPatient().getUserProfile();
		if (up != null) {
			String first = up.getFirstName();
			String last = up.getLastName();
			String middle = up.getMiddleName();
			buffer.append(first);
			if (!StringUtils.isBlank(middle)) {
				middle=middle.substring(0,1);
				buffer.append(" ");
				buffer.append(middle);
				buffer.append(".");
			}
			buffer.append(" ");
			buffer.append(last);
		}
		return buffer.toString();
	}
    

    protected String getPatientUserName() {
        return getPatient().getUserProfile().getUserName();
    }

    /**
     * Determines if the extract type of this controller is enabled
     * @return True if it is enabled, otherwise false.
     * @see #getExtractType()
     */
    public boolean isEnabled() {
        return SessionManagement.getInstance().isExtractEnabled(getRequest(),
            getExtractType());
    }
    /**
     * Determines if the extract type of this controller is partialed
     * @return True if it is partialed, otherwise false.
     * @see #getExtractType()
     */
    public boolean isPartialed() {
        return SessionManagement.getInstance().isExtractPartialed(getRequest(),
            getExtractType());
    }


    /**
     * Determines if a refresh request for the extract type of this
     * controller is currently in progress for the current patient.
     * @return True if a request is in progress, otherwise false.
     * @see #getExtractType()
     * @see #getPatient()
     */
    public boolean getInProgress() {
        return (status != null) && BooleanUtils.isTrue(status.getInProgress());
    }

    /**
     * Returns the date of the last update of the extract data as available
     * in the controller.
     * @return The date of the last extract update.
     * @see #getLastUpdate(PatientExtractStatus)
     */
    public Date getLastUpdate() {
        return getLastUpdate(status);
    }

    /**
     * Returns the properly formatted date of the last update of
     * the extract data as available in the controller.
     * @return The formatted date of the last extract update.
     * @see #getLastUpdate()
     */
    public String getFormattedLastUpdate() {
        Date lastUpdate = getLastUpdate();
    	if (lastUpdate == null) {
            return null;
        }
        return new SimpleDateFormat("MM/dd/yyyy 'at' HH:mm").format(lastUpdate);
    }

    /* (non-Javadoc)
     * @see org.tigris.atlas.beehive.EntityListMaintenancePageFlowController#
     * lookupEntities()
     */
    @Override
    // Note that the extract may have finished loading after we
    // requested the status (last updated date and the inProgress
    // status). So potentially we show the user that an update is
    // in progress when it really has already completed. The
    // chance of this happening is probabaly small and the user
    // can still refresh the page. So, this is an acceptable risk.
    // Unfortunately, we cannot create a service method that
    // returns an object that combines the status and the extract
    // data using Atlas, because we would have to create an Entity.
    // Alternatively, this method could start a new transaction.
    protected final EntityCollectionServiceResponse lookupEntities() {
        EntityCollectionServiceResponse response =
            createExtractServiceResponse(null);
        boolean isDisplayable = isDisplayable();
        if (isDisplayable) {
            PatientExtractStatusServiceResponse statusResponse =
                gov.va.med.mhv.usermgmt.service.delegate.
                ServiceDelegateFactory.createExtractServiceDelegate().
                getPatientStatus(getExtractType(), getPatient());
            if (hasErrorMessages(statusResponse)) {
                copyMessages(response, statusResponse);
            } else {
                PatientExtractStatus status = statusResponse.
                    getPatientExtractStatus();
                final boolean mustLoadExtract = isNewer(status) ||
                    // when the extract was not enabled previously
                    // the extract may not have been loaded yet
                    !wasDisplayableOnLastLoad();
                if (LOG.isDebugEnabled()) {
                    LOG.debug(((mustLoadExtract) ? "L" : "Not l")
                        + "oading extract data (last session update="
                        + getLastUpdate() + "; last extract update="
                        + getLastUpdate(status) + "; was displayable ="
                        + wasDisplayableOnLastLoad() + ")"
                        + " for patient '" + getPatientUserName() + "'");
                }
                if (mustLoadExtract) {
                    response = loadExtract();
                } else {
                    response = createExtractServiceResponse(getEntities());
                }
                if (isUpdateComplete(status)) {
                    addExtractUpdateCompleteMessage();
                }
                setStatus(status);
            }
        }
        Precondition.assertNotNull("response.entities", response.getEntities());
        wasDisplayableOnLastLoad = isDisplayable();
        return response;
    }

    protected EntityCollectionServiceResponse
        createExtractServiceResponse(Collection entities)
    {
        EntityCollectionServiceResponse response =
            createExtractServiceResponse();
        if (entities != null) {
            response.addItems(entities);
        }
        return response;
    }

    /**
     * Yields the new collection instance that can hold the extract
     * service responses.
     * This method must return the same type as is returned fom the
     * {@link #loadExtract()} method. The overriding method should simply
     * create the instance and return it.
     * @return The EntityCollectionServiceResponse that this controller
     * can hold.
     */
    protected abstract EntityCollectionServiceResponse
        createExtractServiceResponse();


    private boolean isDisplayable() {
        return (
        		(displayEvenWhenExtractDisabled()
        		|| isEnabled() 	|| isPartialed())
        		&& patientIsCorrelatedDisplayFlag);
    }

    private boolean wasDisplayableOnLastLoad() {
        return wasDisplayableOnLastLoad;
    }

    /**
     * Determines whether a given PatientExtractStatus is newer than the
     * PatientExtractStatus currently associated with this controller.
     * @param status The PatientExtractStatus to compare
     * @return true when there is no last update date associated with the
     * given status or the controller status, or the date of the given
     * status is strictly before the current controller status.
     * @see #getLastUpdate() The current last update associated with
     * this controller.
     */
    private boolean isNewer(PatientExtractStatus status) {
        Date lastSessionUpdate = getLastUpdate();
        if (lastSessionUpdate == null) {
            return true;
        }
        Date lastExtractUpdate = getLastUpdate(status);
        return ((lastExtractUpdate == null) || lastSessionUpdate.before(
            lastExtractUpdate));
    }

    private void setStatus(PatientExtractStatus status) {
        this.status = (status != null) ? status : createPatientExtractStatus();
    }

    private boolean isUpdateComplete(PatientExtractStatus status) {
        boolean wasInProgress = (this.status != null) &&
            this.status.getInProgress();
        boolean isNowCompleted = (status != null) && !status.getInProgress();
        return wasInProgress && isNowCompleted;
    }


    /**
     * Creates a new PatientExtractStatus instance.
     * @return
     */
    private static PatientExtractStatus createPatientExtractStatus() {
        PatientExtractStatus status = TransferObjectFactory.
            createPatientExtractStatus();
        status.setLastUpdated(null);
        status.setInProgress(false);
        return status;
    }

    /**
     * Determines the last update for a given PatientExtractStatus instance.
     * @param status The status to check.
     * @return The last updated attribute value of the status if it is not null;
     * null otherwise.
     */
    private Date getLastUpdate(PatientExtractStatus status) {
        return (status != null) ? status.getLastUpdated() : null;
    }

    /**
     * Mandates that child page controllers of PHR should log
     * view activities.
     *
     */
    protected abstract void auditView();

    /**
     * Returns the collection of extract records for this extract controller.
     * Note that method replaces the {@link #lookupEntities()} method.
     * In addition it will also handle, determining if the extract can be
     * loaded: is the extract enabled (and if not, should the data be loaded
     * anyway) and is there newer extract data available.
     * @return The collection of entract entity instances.
     */
    protected abstract EntityCollectionServiceResponse loadExtract();

    /**
     * Provides the extract type used by the controller to determine
     * status and enablement of the extract.
     * @returns The extract type.
     */
    protected abstract ExtractType getExtractType();

    /**
     * Provides the name of extract type used by the common page fragments
     * to display the name of the extract type.
     * This is used e.g. to show that the extract is disabled.
     * @returns The name of extract type.
     * @see #getExtractName()
     */
    protected abstract String getExtractName();

    /**
     * Provides the description of extract type used by the common page 
     * fragments to display the description of the extract type.
     * This is e.g. used to show that an extract has been updated
     * The default value is to return the extract name.
     * Override this method to provide a different value.
     * @returns The name of extract type.
     * @see #getExtractName()
     */
    protected String getExtractDescription() {
        return getExtractName();
    }
    

    /**
     * Denotes whether the extract must be loaded (from Evault) even when
     * the extract is disabled.
     * Override this method when the evailable extract data must always
     * be shown.
     * If this method is not overriden, the entity list will be empty.
     * @return Returns whether the extract is loaded from evault. The default
     * value is false;
     * @see #getExtractType()
     */
    protected boolean displayEvenWhenExtractDisabled() {
        return false;
    }

    /**
     * Reloads the patient stored in the session.
     * reloading the patient stored in the session from Evault may be
     * necessary if changes to ICN and/or facilities must be reflected
     * immediately.
     * @see #getPatient();
     */
    protected void refreshPatient() {
        Patient patient = getPatient();
        Precondition.assertNotNull("patient", patient);
        PatientServiceResponse response = ServiceDelegateFactory.
            createEntityMaintenanceServiceDelegate().findByPrimaryKey(patient.
            getPatientPK());
        if (hasErrorMessages(response)) {
            handleMessages(response);
            SessionManagement.getInstance().setPatient(getRequest(), null);
        } else {
            SessionManagement.getInstance().setPatient(getRequest(), patient);
        }
    }

    public void setInPersonAuthenticationUrl(String inPersonAuthenticationUrl) {
        this.inPersonAuthenticationUrl = inPersonAuthenticationUrl;
    }
    
    public String getInPersonAuthenticationUrl() {
        return inPersonAuthenticationUrl;
    }

    /**
     * Creates the URL that links the JSP to the page with information about
     * in-person-authentication.
     * This method must be called BEFORE forwarding to the JSP that needs it,
     * because the response instance will be in invalid state when the JSP is
     * being written and the URL cannot be created at that time
     **/
    protected String createInPersonAuthenticationUrl() {
        PageURL url = PageURL.createPageURL(getRequest(), getResponse());
        url.setPageLabel(IPA_PORTAL_PAGE);
        return getResponse().encodeURL(url.toString());
    }

    protected void addExtractUnavailableMessage() {
        addMessage(PHRMessages.createExtractUnavailableMessage(
            getExtractName()));
    }

    protected void addExtractUpdatesUnavailableMessage() {
        addMessage(PHRMessages.createExtractUpdatesUnavailableMessage(
            getExtractName()));
    }
//    private void addExtractPatientUncorrelatedMessage() {
//        addMessage(PHRMessages.createExtractPatientUnCorrelated(
//            getExtractName()));
//    }

    private void addExtractUpdateCompleteMessage() {
        addMessage(PHRMessages.createExtractUpdateCompleteMessage(
            getExtractName()));
    }

    protected void addMessage(Message message) {
        Messages messages = MessageFactory.createMessages();
        messages.addMessage(message);
        handleMessages(messages);
    }

}
